import Card from './Card.js';
import Text from './Text.js';

export default class Solitaire{
    #step;

    constructor(){
        this.canvasSize = 320;                                  //Taille du canvas
        this.images = ["solitaire.png", "cards.png", "text.png", "popup.png"];   //Images à charger
        this.sounds = ["cardPlace1.wav","cardPlace2.wav","cardPlace3.wav","cardShove1.wav","cardShove2.wav","cardShove3.wav"]; //Sons à charger
        this.nbdeck = 52;                                      //Nombre de carte dans le paquet
        this.timerMax = 10 * 60;                                 //5 * 60sec => 5 min
        this.durationMove = 0.3;
        this.durationDelay = 0.1;
        this.margeYColumn = 10;
        this.#step = "LOADING";
        this.ready = false;
        //Coordonnée et options de toutes les zones cliquables
        this.zoneList = [{x:279, y:15, w:26, h:26, name:"replay", isGameOver:false},
                        {x:124, y:199, w:72, h:16, name:"replay", isGameOver:true},
                        {x:271, y:249, w:32, h:48, name:"deck"},
                        {x:231, y:249, w:32, h:48, name:"discard"},
                        {x:23, y:249, w:32, h:48, name:"stock0", sign:0},
                        {x:68, y:249, w:32, h:48, name:"stock1", sign:2},
                        {x:113, y:249, w:32, h:48, name:"stock2", sign:1},
                        {x:158, y:249, w:32, h:48, name:"stock3", sign:3},
                        {x:24, y:55, w:32, h:48, name:"column0", id:0},
                        {x:64, y:55, w:32, h:48, name:"column1", id:1},
                        {x:104, y:55, w:32, h:48, name:"column2", id:2},
                        {x:144, y:55, w:32, h:48, name:"column3", id:3},
                        {x:184, y:55, w:32, h:48, name:"column4", id:4},
                        {x:224, y:55, w:32, h:48, name:"column5", id:5},
                        {x:264, y:55, w:32, h:48, name:"column6", id:6}
                    ];
    }

    handleLoaded() { 
        let firstSoundName = this.sounds[0].split(".")[0];
        if(!APP.sounds[firstSoundName]){ APP.loadSound(this.sounds); return;}
        this.#step = "NEW";
    }

    update(dt){
        //Mise à jour du timer
        this.updateTimer(dt);
        //Initialisation d'une nouvelle partie
        if(this.#step === "NEW"){ this.newParty(); this.ready = true; }
        //Si des cartes sont en mouvement
        else if(this.#step === "MOVE"){            
            if(this.movement.length === 0){ 
                const columnReturn = this.checkColumnNoFlipped();
                if(this.discard.length === 0 && this.deck.length > 0){ this.clicdiscard(); this.moveCard(this.deck[0], 'deck', 'discard', true); }  
                else if(columnReturn){ this.moveCard(columnReturn.card, columnReturn.column, columnReturn.column, true); }
                else if(!columnReturn && this.checkAllClear()){ this.moveMoreLittleCard(); }
                else{ this.#step = "WAIT"; }                
            }else{
                this.updateMove(dt);
            }
        }
        //Si le joueur sélection une colonne ou la défausse
        else if(this.#step === "SELECT"){
            APP.isUpdated = true; this.#step = "WAIT";
        }
        //Si le jeu est en situation d'attente d'action du joueur
        else if(this.#step === "WAIT"){ }
        //GAMEOVER
        if(this.isGameOver && this.#step !== "WAIT"){ this.#step = "WAIT"; }
    }

    render(ctx) {
        ctx.drawImage(APP.images['solitaire'], 0, 0);    //Fond du jeu
        this.drawScore(ctx);
        this.drawTimer(ctx);
        this.drawSelected(ctx);
        this.drawCards(ctx);
        this.drawPopup(ctx);
    }

    // GETTERS / SETTERS =========================
    get isGameOver() { 
         // La partie prend fin si le timer atteint 0
         // Ou si toutes les cartes se trouvent dans les stocks
        return this.ready && (this.timer <= 0 || this.totalStock === this.nbdeck);
    }
    get totalStock() {
        let total = 0;
        for(let s of this.stock){ total += s.length; }   // On ajoute la longueur de chaque stock au total
        return total;
    }
    getZone(name){
        //Retourne les information de la zone demandée (en incluant les cartes qui la compose)
        for(let z of this.zoneList){
            if(z.name === name){ 
                if(name.includes('column')){ z.cards = this.column[name.substring(name.length-1)]; }
                else if(name.includes('stock')){ z.cards = this.stock[name.substring(name.length-1)]; }   
                else{ z.cards = this[name]; } 
                return z; 
            }
        }
        return null;
    }
    getFixColumnY(zone){
        if(!zone.name.includes('column')) return 0;
        let prevCardZone = 0;        
        for(let item of this.movement){ if(item.ze === zone){ prevCardZone++; } }
        return (prevCardZone + zone.cards.length) * this.margeYColumn;
    }
    checkColumnNoFlipped(){
        for(let i in this.column){
            if(this.column[i].length === 0){ continue; }
            let lastCard = this.column[i][this.column[i].length-1];
            if(!lastCard.isFlipped){ return {card:lastCard, column:'column'+i}; }
        }
        return false;
    }
    checkAllClear(){
        if(this.deck.length > 0 || this.discard.length > 0 || this.totalStock === this.nbdeck){ return false; }
        for(let col of this.column){
            for(let card of col){
                if(!card.isFlipped){ return false; }
            }
        }
        return true;
    }
    moveMoreLittleCard(){
        let moreLittle = {card:null, column:null};
        for(let iCol in this.column){
            for(let card of this.column[iCol]){
                if(!moreLittle.card || moreLittle.card.number > card.number){ 
                    moreLittle.card = card; 
                    moreLittle.column = 'column'+iCol; 
                }
            }
        }
        let stock = 'stock'+moreLittle.card.sign;
        if(moreLittle.card.sign === 1){ stock = 'stock2'; }
        if(moreLittle.card.sign === 2){ stock = 'stock1'; }
        this.moveCard(moreLittle.card, moreLittle.column, stock, true);
        return;
    }


    // METHODS ===================================
    updateTimer(dt){
        if(this.isGameOver){ return; }
        if(Math.floor(this.timer) !== Math.floor(this.timer - dt)){ APP.isUpdated = true; }
        this.timer = Math.max(0, this.timer - dt);
    }
    getTimer(){
        let min = Math.floor(Math.floor(this.timer)/60);
        let sec = ("00" + Math.floor(this.timer)%60);
        sec = sec.substring(sec.length-2);
        return min +":"+ sec;
    }

    // METHODS STEP NEW ==========================
    newParty(){                                 //Se déclenche au début de la partie
        this.score = 0;                         //Score du joueur
        this.timer = this.timerMax;             //Temps restant avant la fin de la partie
        this.deck = [];                         //Liste des crates dans le deck
        this.discard = [];                      //Liste de carte dans la defausse
        this.createDeck();                      //Création des cartes du paquet + Mélange
        this.column = [[],[],[],[],[],[],[]];   //Liste des colonnes
        this.stock = [[],[],[],[]];             //Liste des stocks
        this.movement = [];                     //Liste des cartes en mouvement
        this.selected = null;                   //Désigne la zone sélectionnée (pour afficher une flèche)
        this.distribCards();                    //Distribution des cartes pour chaque colonne
    }
    createDeck(){
        const deck = this.getZone('deck');
        for(let index=0; index<this.nbdeck; index++){
            let card = new Card(index);
            card.x = deck.x + (card.width/2);
            card.y = deck.y + (card.height/2);
            this.deck.push(card);
        }
        this.deck = this.deck.sort(() => Math.random() - 0.5); 
    }
    distribCards(){
        let delay = 0;
        //Distributions des cartes sur les différentes colonnes
        for(let row=0; row<this.column.length; row++){
            //On distribut autant de cartes par colonne que sa position
            for(let col=0; col<this.column.length; col++){
                if(row > col){ continue; }
                this.moveCard(this.deck[0], 'deck', 'column'+col, false, delay);
                delay += this.durationDelay; 
            }
        }
    }

    // METHODS STEP MOVE =========================
    moveCard(card, zoneStart, zoneEnd, flip, delay=0){
        const zs = this.getZone(zoneStart);
        const ze = this.getZone(zoneEnd);
        let fixY = this.getFixColumnY(ze);
        if(zs === ze){ fixY -= this.margeYColumn; }
        this.movement.push({card, sX:card.x, sY:card.y, eX:ze.x+16, eY:ze.y+24+fixY, timer:0, delay, flip, ze});
        this.#step = "MOVE";
        //Suppresion de la zone de départ
        let index = zs.cards.indexOf(card);
        if(index >= 0){ zs.cards.splice(index,1); }
    }

    updateMove(dt){
        let listFinish = [];
        for(let item of this.movement){
            //On attend que le délai demandé soit atteint avant de déplacer la carte
            if(item.delay>0){ item.delay -= dt; continue; }
            //Activation du son de la carte
            if(item.timer === 0){
                try{
                    let r = Math.ceil(Math.random()*3); //génère un entier entre 1 et 3 inclus
                    if(item.card.isFlipped !== item.flip){ APP.sounds['cardShove'+r].play(); }
                    else{ APP.sounds['cardPlace'+r].play(); }
                }catch(error){}
            }
            //On l'animation de déplacement de la carte (en limitant le timer au max)
            item.timer += dt;
            item.timer = Math.min(item.timer, this.timerMax);
            //Déplacement de la carte
            const ease = APP.easeInOutQuad(item.timer, this.durationMove);
            item.card.x = item.sX + (item.eX - item.sX) * ease;
            item.card.y = item.sY + (item.eY - item.sY) * ease;   
            //Flip de la carte (si demandé)
            if(item.card.isFlipped !== item.flip || item.card.scaleX !== 1){
                item.card.scaleX = Math.abs(ease * 2 - 1); 
                if(item.timer >= this.durationMove/2){ item.card.isFlipped = item.flip; }
            }
            //Quand l'animation est terminée, on déplace la carte dans la zone prévue
            //Et on lui applique les valeurs attendues
            if(item.timer >= this.durationMove){
                item.card.x = item.eX;
                item.card.y = item.eY;
                item.card.scaleX = 1;
                item.card.scaleY = 1;
                listFinish.push(item);
                //Pour le deck on ajoute les carte au début du tableau, sinon c'est à la fin
                if(item.ze.name === 'deck'){ item.ze.cards.unshift(item.card); }
                else{ item.ze.cards.push(item.card); }
                continue;
            }
        }
        //Suppression de la liste des élément ayant fini leur mouvement
        while(listFinish.length > 0){
            let index = this.movement.indexOf(listFinish[0]);
            this.movement.splice(index,1);
            this.listFinish.shift();
        }
    }

    // METHODS DRAW ==============================
    drawScore(ctx) {
        const img = Text.img(this.score);
        ctx.save();
        ctx.translate(77-img.width, 22);
        ctx.drawImage(img, 0, 0);
        ctx.restore();
    }
    drawTimer(ctx){
        const img = Text.img(this.getTimer());
        ctx.save();
        ctx.translate(160-Math.floor(img.width/2), 33);
        ctx.drawImage(img, 0, 0);
        ctx.restore();
    }
    drawSelected(ctx){
        if(!this.selected) return;
        ctx.save();
        ctx.translate(this.selected.x, this.selected.y);
        ctx.drawImage(APP.images['solitaire'], 0, 320, 12, 8, 10, -8, 12, 8);
        ctx.restore();
    }
    drawCards(ctx) {
        for(let card of this.deck){ card.draw(ctx); }
        for(let card of this.discard){ card.draw(ctx); }
        for(let col of this.column){ for(let card of col){ card.draw(ctx); } }
        for(let s of this.stock){ for(let card of s){ card.draw(ctx); } }
        for(let item of this.movement){ item.card.draw(ctx); }
    }
    drawPopup(ctx) {
        if(!this.isGameOver) return;          // Si la partie n'est pas terminée, on ne dessine pas la popup
        let x = 89;
        let y = 110;
        let imgY = this.timer <= 0 ? 106 : 0; 
        const img = APP.images["popup"];
        ctx.drawImage(img, 0, imgY, 142, 106, x, y, 142, 106);  
    }

    // Méthodes Evenements
    handleClick(x, y) {
        //On ne prend aucun clic en compte si le jeu n'est pas attente d'action du joueur
        if(this.#step !== "WAIT") return; 
        let zone = this.clicZone(x, y);
        if(!zone) return;            // Aucun bouton n'est dans la zone de clic
        if(zone.name === "replay" && zone.isGameOver === this.isGameOver){ this.#step = "NEW"; return; }
        if(this.isGameOver) return;
        let name = zone.name;
        if(name.includes('column') || name.includes('stock')){ name = name.substring(0, name.length-1); }
        try{ this['clic' + name](zone); }catch(error){}
    }

    clicZone(x, y) {
        for(let zone of this.zoneList){
            let fixY = 0;
            if(zone.name.includes('column')){ fixY = (zone.cards.length-1) * this.margeYColumn; }
            // Si le clic se trouve dans la zone du bouton, on le renvoit
            if(x >= zone.x && x <= zone.x + zone.w && y >= zone.y && y <= zone.y + zone.h + fixY){ return zone; }
        }
        return false;
    }
    
    clicdeck(zone){
        //S'il y a au moins une carte dans le deck, on pioche une nouvelle carte
        if(this.deck.length > 0){
            this.score++;
            this.clicdiscard();
            this.moveCard(this.deck[0], zone.name, 'discard', true); 
            return;
        }
        //Sinon, on remet les cartes de la défausse dans le deck
        let delay = this.durationDelay * (this.discard.length);
        while(this.discard.length > 0){
            this.moveCard(this.discard[0], 'discard', zone.name, false, delay);
            delay -= this.durationDelay; 
        }
    }
    clicdiscard(){
        this.selected = this.getZone('discard');
        this.#step = 'SELECT';
    }
    cliccolumn(zone){
        if(this.selected != zone){
            //On prend la dernière carte de la colonne en cours
            let columnCard = null;
            if(zone.cards.length>0){ columnCard = zone.cards[zone.cards.length-1]; }
            //On passe toute les cartes en revue de la zone sélectionnée en remontant dans la liste
            //Pour pouvoir déplacer des suites entières
            //Si la suite n'est plus logique, on considère que le déplacement est impossible
            let prevCard = [];
            for(let i=this.selected.cards.length-1; i>=0; i--){
                let card = this.selected.cards[i];
                let pcard = null;
                if(prevCard.lentgh > 0){ pcard = prevCard[0]; }
                //On ne prend pas en compte es carte face cachée evidemment
                if(!card.isFlipped){ break; }
                if(pcard && (card.color === pcard.color || card.number != pcard.number+1)){ break; }
                //Si la couleur de la carte est différente et que le nombre == à celle de la colonne + 1
                //Ou si un Roi souhaite aller sur un emplacement vide
                //Le déplacement peut s'effectuer
                if((!columnCard && card.number === 12) || (columnCard && columnCard.color !== card.color && columnCard.number === card.number+1)){      
                    this.score++;   
                    this.moveCard(card, this.selected.name, zone.name, true);  
                    let delay = 0      
                    for(let c of prevCard){ 
                        delay += this.durationDelay;
                        this.moveCard(c, this.selected.name, zone.name, true, delay);                        
                    }                    
                    return;
                }
                //On ne verifie pas les précédente carte pour la défausse
                if(this.selected.name === 'discard'){ break; }
                prevCard.unshift(card);
            }            
        }
        //Sinon on sélectionne la colonne
        this.selected = zone; this.#step = 'SELECT';
    }

    clicstock(zone){
        if(!this.selected) return;
        let s = this.getZone(zone.name);
        //On vérifie si la dernière carte de la zone sélectionné peut aller dans le stock
        let card = this.selected.cards[this.selected.cards.length-1];
        //Si le symbole de la carte est identique à celui du stock et que le nombre == au nombre de cartes dans le stock
        //Le déplacement peut s'effectuer
        if(zone.cards.length === card.number && zone.sign === card.sign){  
            this.score++;       
            this.moveCard(card, this.selected.name, zone.name, true);  
        }    
    }

}